<?php
/**
 * Reusable User Info Component
 *
 * Provides a consistent user information display across all block templates.
 *
 * @since 4.2.0
 */

// Ensure we have the required data
if (!isset($user) || !is_object($user) || !isset($user->ID)) {
    return;
}

// Get component options
$options = $options ?? [];
$defaults = [
    'show_name' => true,
    'name_format' => 'display_name',
    'name_heading' => 'h3',
    'show_bio' => true,
    'bio_limit' => 100,
    'show_role' => false,
    'show_registration_date' => false,
    'show_contact' => false,
    'contact_options' => [],
    'class' => '',
];

$options = wp_parse_args($options, $defaults);

// Build wrapper classes
$wrapper_classes = ['wpuf-user-info-component'];
if ($options['class']) {
    $wrapper_classes[] = $options['class'];
}

// Start building the component HTML
$component_html = '';

// User name
if ($options['show_name']) {
    $name = wpuf_get_user_display_name($user, $options['name_format']);
    $heading_tag = $options['name_heading'];

    $component_html .= sprintf(
        '<%1$s class="wpuf-user-name">%2$s</%1$s>',
        esc_attr($heading_tag),
        $name
    );
}

// User role
if ($options['show_role']) {
    $role_display = wpuf_get_user_role_display($user);
    if (!empty($role_display)) {
        $component_html .= sprintf(
            '<div class="wpuf-user-role">%s</div>',
            esc_html($role_display)
        );
    }
}

// Registration date
if ($options['show_registration_date']) {
    $reg_date = wpuf_get_user_registration_date($user);
    if (!empty($reg_date)) {
        $component_html .= sprintf(
            '<div class="wpuf-user-registration-date">
                <span class="wpuf-label">%s:</span>
                <span class="wpuf-value">%s</span>
            </div>',
            esc_html__('Member since', 'wpuf-pro'),
            esc_html($reg_date)
        );
    }
}

// User bio
if ($options['show_bio']) {
    $bio_html = wpuf_get_user_bio($user, [
        'character_limit' => $options['bio_limit'],
        'show_read_more' => true,
    ]);

    if (!empty($bio_html)) {
        $component_html .= $bio_html;
    }
}

// Contact information
if ($options['show_contact']) {
    $contact_html = wpuf_get_user_contact_info($user, $options['contact_options']);
    if (!empty($contact_html)) {
        $component_html .= $contact_html;
    }
}

// Output the component
if (!empty($component_html)) {
    printf(
        '<div class="%s">%s</div>',
        esc_attr(implode(' ', $wrapper_classes)),
        $component_html
    );
}
