<?php
/**
 * User Avatar Block Template
 *
 * @since 4.2.0
 *
 * @var WP_User $user User object
 * @var array $attributes Block attributes
 * @var string $content Block content
 * @var WP_Block $block Block instance
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! $user ) {
    return;
}

// Get dynamic styling from block attributes
$styling = wpuf_get_avatar_styling( $attributes, $user );

// Get avatar URL and display size
$avatar_size = isset( $attributes['avatarSize'] ) ? $attributes['avatarSize'] : 'medium';
$custom_size = isset( $attributes['customSize'] ) ? intval( $attributes['customSize'] ) : 150;
$fallback_type = isset( $attributes['fallbackType'] ) ? $attributes['fallbackType'] : 'initials';
$avatar_shape = isset( $attributes['avatarShape'] ) ? $attributes['avatarShape'] : 'circle';

// Calculate display size including custom size
$display_size = wpuf_ud_get_avatar_display_size( $avatar_size, $custom_size );

// Get avatar data with custom profile photo priority
$avatar_data = wpuf_ud_get_block_avatar_data( $user, $display_size, $fallback_type );
$avatar_url = $avatar_data['url'];
$avatar_type = $avatar_data['type'];
$avatar_alt = $avatar_data['alt'];

$initials = wpuf_ud_get_user_initials( $user );

// Get size and shape classes
$size_class = 'wpuf-avatar--' . $avatar_size;
$shape_class = 'wpuf-avatar--' . $avatar_shape;

// Check if current user can edit this profile
$can_edit = false;
if ( class_exists( 'WPUF\UserDirectory\Blocks' ) ) {
    $blocks_instance = new WPUF\UserDirectory\Blocks();
    $can_edit = $blocks_instance->can_user_edit_profile( $user );
}

// Get block wrapper attributes
$wrapper_attributes = wpuf_get_block_wrapper_attributes( $attributes, 'avatar', [
    'class' => 'wpuf-user-avatar',
    'data-user-id' => $user->ID,
] );

// Add custom size CSS variable if needed
$custom_style = '';
if ( $avatar_size === 'custom' ) {
    $custom_style = '--wpuf-avatar-size: ' . $custom_size . 'px;';
}

?>
<div <?php echo $wrapper_attributes; ?>>
    <div class="wpuf-avatar-container" <?php echo $custom_style ? 'style="' . esc_attr( $custom_style ) . '"' : ''; ?>>
        <?php if ( $avatar_type === 'custom' || $avatar_type === 'gravatar' ) : ?>
            <!-- Show avatar image with fallback -->
            <img src="<?php echo esc_url( $avatar_url ); ?>"
                 alt="<?php echo esc_attr( $avatar_alt ); ?>"
                 class="wpuf-avatar--image <?php echo esc_attr( $size_class . ' ' . $shape_class ); ?>"
                 onerror="this.style.display='none'; this.nextElementSibling.style.display='flex';" />
            <!-- Fallback initials (hidden by default) -->
            <div class="wpuf-avatar--initials <?php echo esc_attr( $size_class . ' ' . $shape_class ); ?>" style="display: none;">
                <span class="wpuf-avatar-initials-text"><?php echo esc_html( $initials ); ?></span>
            </div>
        <?php else : ?>
            <!-- Show initials (no image available) -->
            <div class="wpuf-avatar--initials <?php echo esc_attr( $size_class . ' ' . $shape_class ); ?>">
                <span class="wpuf-avatar-initials-text"><?php echo esc_html( $initials ); ?></span>
            </div>
        <?php endif; ?>
        
        <?php
        // Only show upload button in User Profile block, not in User Directory block
        $is_profile_mode = isset( $is_profile_mode ) ? $is_profile_mode : false;
        if ( $can_edit && $is_profile_mode ) :
        ?>
            <!-- Upload Button -->
            <button type="button" 
                    class="wpuf-avatar-upload-btn wpuf-absolute wpuf-inset-0 wpuf-flex wpuf-items-center wpuf-justify-center wpuf-bg-black wpuf-bg-opacity-50 wpuf-text-white wpuf-text-sm wpuf-font-medium wpuf-rounded-full wpuf-opacity-0 wpuf-transition-opacity wpuf-duration-200 wpuf-cursor-pointer hover:wpuf-bg-opacity-70 focus:wpuf-outline-none focus:wpuf-ring-2 focus:wpuf-ring-green-500 focus:wpuf-ring-opacity-50" 
                    onclick="wpufOpenAvatarUpload(<?php echo esc_attr( $user->ID ); ?>)"
                    title="<?php esc_attr_e( 'Upload Profile Photo', 'wpuf-pro' ); ?>"
                    aria-label="<?php esc_attr_e( 'Upload Profile Photo', 'wpuf-pro' ); ?>">
                <span class="wpuf-avatar-upload-text"><?php esc_html_e( 'Upload', 'wpuf-pro' ); ?></span>
            </button>
        <?php endif; ?>
    </div>
</div>
