<?php
namespace AvasElements\Modules\SearchPlus\Widgets;

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Css_Filter;
use Elementor\Core\Kits\Documents\Tabs\Global_Colors;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Repeater;
use Elementor\Core\Kits\Documents\Tabs\Global_Typography;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Icons;
use AvasElements\TX_Helper;


if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class SearchPlus extends Widget_Base {

    public function get_name() {
        return 'avas-search-plus';
    }

    public function get_title() {
        return esc_html__( 'Avas Search Plus', 'avas-core' );
    }

    public function get_icon() {
        return 'eicon-search-bold';
    }

    public function get_categories() {
        return [ 'avas-elements' ];
    }
    public function get_keywords() {
        return [ 'search', 'ajax search' ];
    }
    public function get_script_depends() {
        return ['tx-search-plus'];
    }
    public function get_style_depends() {
        return ['tx-search-plus'];
    }

    /**
     * Fetch all public posts (any CPT) for the exclude dropdown.
     *
     * @return array [ID => "Post Title (post_type)"]
     */
    private function get_all_posts_options() {
        $options = [];
        // get all public post types
        $post_types = get_post_types( ['public' => true], 'names' );

        // fetch all posts in those types
        $all = get_posts( [
            'post_type'      => $post_types,
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
        ] );

        foreach ( $all as $p ) {
            $options[ $p->ID ] = sprintf(
                '%s (%s)',
                $p->post_title,
                $p->post_type
            );
        }

        return $options;
    }

    public function add_control_ajax_search_img_size() {

        $intermediate_image_sizes = [];

        foreach ( get_intermediate_image_sizes() as $key=>$value ) {
            $intermediate_image_sizes[$value] = $value;
        }

        $this->add_control(
            'ajax_search_img_size',
            [
                'label' => esc_html__( 'Image Crop', 'avas-core' ),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => $intermediate_image_sizes,
                'default' => 'thumbnail',
            ]
        );
    }

    public function add_control_search_query() {

        $search_post_types = TX_Helper::get_custom_types_of( 'post', false );
        $options = [ 'all' => esc_html__( 'All', 'avas-core' ) ];

        foreach ( $search_post_types as $slug => $label ) {
            $options[$slug] = $label;
        }

        $this->add_control(
            'search_query',
            [
                'label' => esc_html__( 'Select Query', 'avas-core' ),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => $options,
                'default' => 'all',
            ]
        );
    }

    public function add_control_all_cat_text() {
        // "All" label used in the select
        $this->add_control(
            'all_cat_text',
            [
                'label'   => esc_html__( 'All Text', 'avas-core' ),
                'type'    => Controls_Manager::TEXT,
                'dynamic' => [ 'active' => true ],
                'default' => esc_html__( 'All Categories', 'avas-core' ),
                'condition' => [
                    'search_query!'   => 'all',
                    'select_category' => 'yes',
                ],
            ]
        );

        // Build one taxonomy control per post type
        $search_post_types = TX_Helper::get_custom_types_of( 'post', false );

        foreach ( $search_post_types as $slug => $label ) {
            $tax_objects = get_object_taxonomies( $slug, 'objects' );
            if ( empty( $tax_objects ) ) {
                continue;
            }

            $options = [];
            foreach ( $tax_objects as $tax ) {
                $options[ $tax->name ] = $tax->label;
            }

            $this->add_control(
                'query_taxonomy_' . $slug,
                [
                    'label'       => sprintf( esc_html__( '%s Taxonomies', 'avas-core' ), $label ),
                    'type'        => Controls_Manager::SELECT2,
                    'options'     => $options,
                    'multiple'    => true,
                    'label_block' => true,
                    'condition'   => [
                        'select_category' => 'yes',
                        'search_query'    => $slug, // only show for this CPT
                    ],
                ]
            );
        }
    }

    protected function register_controls() {
        
        // Section: Search -----------
        $this->start_controls_section(
            'section_search',
            [
                'label' => esc_html__( 'Search', 'avas-core' ),
            ]
        );

        $this->add_control_search_query();

        $this->add_control(
            'select_category',
            [
                'label' => esc_html__( 'Enable Taxonomy Filter', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'separator' => 'before',
                'condition' => [
                    'search_query!' => 'all'
                ]
            ]
        );

        $this->add_control_all_cat_text();

        $this->add_control(
            'ajax_search',
            [
                'label' => esc_html__( 'Enable Ajax Search', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'number_of_results',
            [
                'label' => __( 'Number of Results', 'avas-core' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 1,
                'step' => 1,
                'default' => 5,
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );


        $this->add_control( 'exclude_posts', [
            'label'       => esc_html__( 'Exclude Posts', 'avas-core' ),
            'type'        => Controls_Manager::SELECT2,
            'options'     => $this->get_all_posts_options(),
            'multiple'    => true,
            'label_block' => true,
            'condition'   => [
                'ajax_search' => 'yes',
            ],
        ] );

        // $this->add_control(
        //     'exclude_titles',
        //     [
        //         'label'       => esc_html__( 'Exclude by Title', 'avas-core' ),
        //         'type'        => Controls_Manager::TEXT,
        //         'description' => esc_html__( 'Comma-separated post titles to exclude', 'avas-core' ),
        //         'input_type'  => 'text',
        //         'default'     => '',
        //         'condition' => [
        //             'ajax_search' => 'yes'
        //         ]
        //     ]
        // );

        if ( current_user_can( 'administrator' ) ) {
            $this->add_control(
                'ajax_show_ps_pt',
                [
                    'label' => esc_html__( 'Show Password Protected', 'avas-core' ),
                    'type' => Controls_Manager::SWITCHER,
                    'description' => esc_html__( 'Only for users with capability to read private posts', 'avas-core' ),
                    'condition' => [
                        'ajax_search' => 'yes'
                    ],
                    'render_type' => 'template',
                ]
            );
        }   

        if ( current_user_can( 'administrator' ) ) {
            $this->add_control(
                'show_attachments',
                [
                    'label' => esc_html__( 'Show Attachments', 'avas-core' ),
                    'description' => esc_html__( 'Include Media Files in Search Results', 'avas-core' ),
                    'type' => Controls_Manager::SWITCHER,
                    'condition' => [
                        'ajax_search' => 'yes'
                    ],
                    'render_type' => 'template',
                ]
            );
        }

        $this->add_control(
            'ajax_search_link_target',
            [
                'label' => esc_html__( 'Open Link in New Tab', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_ajax_thumbnails',
            [
                'label' => esc_html__( 'Show Ajax Thumbnails', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'exclude_posts_without_thumbnail',
            [
                'label' => esc_html__( 'Exclude Results without Thumbnails', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes',
                    'show_ajax_thumbnails' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_view_result_btn',
            [
                'label' => esc_html__( 'Show View Results Button', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'view_result_text',
            [
                'label' => esc_html__( 'View Results', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__( 'View Results', 'avas-core' ),
                'condition' => [
                    'show_view_result_btn' => 'yes',
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_description',
            [
                'label' => esc_html__( 'Show Description', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'number_of_words_in_excerpt',
            [
                'label' => __( 'Description Number of Words', 'avas-core' ),
                'type' => Controls_Manager::NUMBER,
                'min' => 5,
                'step' => 1,
                'default' => 30,
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes',
                    'show_description' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_product_price',
            [
                'label' => esc_html__( 'Show Product Price', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'render_type' => 'template',
                'condition' => [
                    'ajax_search' => 'yes',
                    'search_query' => ['all', 'product', 'pro-product']
                ]
            ]
        );

        $this->add_control(
            'no_results_text',
            [
                'label' => esc_html__( 'No Resulsts Text', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__( 'No Results Found', 'avas-core' ),
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'search_placeholder',
            [
                'label' => esc_html__( 'Placeholder', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__( 'Search...', 'avas-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'search_aria_label',
            [
                'label' => esc_html__( 'Aria Label', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__( 'Search', 'avas-core' ),
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'search_btn',
            [
                'label' => esc_html__( 'Button', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'search_btn_style',
            [
                'label' => esc_html__( 'Style', 'avas-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'inner',
                'options' => [
                    'inner' => esc_html__( 'Inner', 'avas-core' ),
                    'outer' => esc_html__( 'Outer', 'avas-core' ),
                ],
                'prefix_class' => 'tx-search-form-style-',
                'render_type' => 'template',
                'condition' => [
                    'search_btn' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'search_btn_disable_click',
            [
                'label' => esc_html__( 'Disable Button Click', 'avas-core' ),
                'type' => Controls_Manager::SWITCHER,
                'condition' => [
                    'search_btn_style' => 'inner',
                    'search_btn' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'search_btn_type',
            [
                'label' => esc_html__( 'Type', 'avas-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'text' => esc_html__( 'Text', 'avas-core' ),
                    'icon' => esc_html__( 'Icon', 'avas-core' ),
                ],
                'render_type' => 'template',
                'condition' => [
                    'search_btn' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'search_btn_text',
            [
                'label' => esc_html__( 'Text', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => 'Go',
                'condition' => [
                    'search_btn_type' => 'text',
                    'search_btn' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'search_btn_icon',
            [
                'label' => esc_html__( 'Select Icon', 'avas-core' ),
                'type' => Controls_Manager::ICONS,
                'skin' => 'inline',
                'label_block' => false,
                'default' => [
                    'value' => 'fas fa-search',
                    'library' => 'fa-solid',
                ],
                'condition' => [
                    'search_btn_type' => 'icon',
                    'search_btn' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        // Ajax Pagination
        $this->start_controls_section(
            'section_ajax_search_pagination',
            [
                'label' => esc_html__( 'Ajax Pagination', 'avas-core' ),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'pagination_load_more_text',
            [
                'label' => esc_html__( 'Load More Text', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => 'Load More'
            ]
        );

        $this->add_control(
            'pagination_finish_text',
            [
                'label' => esc_html__( 'Finish Text', 'avas-core' ),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => 'All caught up'
            ]
        );

        $this->add_control(
            'pagination_animation',
            [
                'label' => esc_html__( 'Select Animation', 'avas-core' ),
                'type' => Controls_Manager::SELECT,
                'default' => 'loader-1',
                'options' => [
                    'none' => esc_html__( 'None', 'avas-core' ),
                    'loader-1' => esc_html__( 'Loader 1', 'avas-core' ),
                    'loader-2' => esc_html__( 'Loader 2', 'avas-core' ),
                    'loader-3' => esc_html__( 'Loader 3', 'avas-core' ),
                ]
            ]
        );

        $this->add_control(
            'pagination_align',
            [
                'label' => esc_html__( 'Alignment', 'avas-core' ),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left'    => [
                        'title' => esc_html__( 'Left', 'avas-core' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'avas-core' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'avas-core' ),
                        'icon' => 'eicon-text-align-right',
                    ]
                ],
                'default' => 'center',
                'prefix_class' => 'tx-ajax-search-pagination-',
                'render_type' => 'template',
                'separator' => 'before',
            ]
        );

        $this->end_controls_section();

        // Styles
        // Section: Input ------------
        $this->start_controls_section(
            'section_style_input',
            [
                'label' => esc_html__( 'Input', 'avas-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->start_controls_tabs( 'tabs_input_colors' );

        $this->start_controls_tab(
            'tab_input_normal_colors',
            [
                'label' => esc_html__( 'Normal', 'avas-core' ),
            ]
        );

        $this->add_control(
            'input_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_bg_color',
            [
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_placeholder_color',
            [
                'label' => esc_html__( 'Placeholder Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#9e9e9e',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input::-webkit-input-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tx-search-form-input:-ms-input-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tx-search-form-input::-moz-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tx-search-form-input:-moz-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tx-search-form-input::placeholder' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_border_color',
            [
                'label' => esc_html__( 'Border Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'border-color: {{VALUE}};'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'input_box_shadow',
                'selector' => '{{WRAPPER}} .tx-search-form-input-wrap'
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_input_focus_colors',
            [
                'label' => esc_html__( 'Focus', 'avas-core' ),
            ]
        );

        $this->add_control(
            'input_focus_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_focus_bg_color',
            [
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_focus_placeholder_color',
            [
                'label' => esc_html__( 'Placeholder Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#9e9e9e',
                'selectors' => [
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input::-webkit-input-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input:-ms-input-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input::-moz-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input:-moz-placeholder' => 'color: {{VALUE}};',
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input::placeholder' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'input_focus_border_color',
            [
                'label' => esc_html__( 'Border Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'input_focus_box_shadow',
                'selector' => '{{WRAPPER}}.tx-search-form-input-focus .tx-search-form-input-wrap'
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'input_typography_divider',
            [
                'type' => Controls_Manager::DIVIDER,
                'style' => 'thick',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'input_typography',
                'selector' => '{{WRAPPER}} .tx-search-form-input, {{WRAPPER}} .tx-category-select-wrap, {{WRAPPER}} .tx-category-select',
            ]
        );

        $this->add_responsive_control(
            'input_align',
            [
                'label' => esc_html__( 'Alignment', 'avas-core' ),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'default' => 'left',
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'avas-core' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'avas-core' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'avas-core' ),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'text-align: {{VALUE}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'input_border_size',
            [
                'label' => esc_html__( 'Border Size', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 1,
                    'right' => 1,
                    'bottom' => 1,
                    'left' => 1,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'input_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                ],
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'border-radius: 0 0 {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'separator' => 'after',
            ]
        );

        $this->add_responsive_control(
            'input_padding',
            [
                'label' => esc_html__( 'Padding', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 15,
                    'right' => 15,
                    'bottom' => 15,
                    'left' => 15,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-input' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .tx-category-select-wrap::before' => 'right: {{RIGHT}}{{UNIT}};',
                    '{{WRAPPER}} .tx-category-select' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
            ]
        );

        $this->end_controls_section();

        // Styles
        // Section: Select ------------
        $this->start_controls_section(
            'section_style_select',
            [
                'label' => esc_html__( 'Taxonomy Filter', 'avas-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'select_category' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'select_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#333333',
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select-wrap' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tx-category-select' => 'color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'select_bg_color',
            [
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'select_border_color',
            [
                'label' => esc_html__( 'Border Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select' => 'border-color: {{VALUE}};'
                ],
            ]
        );

        $this->add_control(
            'select_border_size',
            [
                'label' => esc_html__( 'Border Size', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 1,
                    'right' => 1,
                    'bottom' => 1,
                    'left' => 1,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'select_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                ],
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}} !important;',
                    '{{WRAPPER}} .tx-category-select-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'after',
            ]
        );

        $this->add_responsive_control(
            'select_width',
            [
                'label' => esc_html__( 'Select Width', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 400,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 230,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select-wrap' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'options_heading',
            [
                'label' => esc_html__( 'Options', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'option_font_size',
            [
                'label' => esc_html__( 'Font Size', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 25,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 12,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select option' => 'font-size: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'optgroup_heading',
            [
                'label' => esc_html__( 'Options Group', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'optgroup_font_size',
            [
                'label' => esc_html__( 'Font Size', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 25,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 12,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-category-select optgroup' => 'font-size: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->end_controls_section();

        // Styles
        // Section: Button ------------
        $this->start_controls_section(
            'section_style_btn',
            [
                'label' => esc_html__( 'Button', 'avas-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'search_btn' => 'yes',
                ],
            ]
        );

        $this->start_controls_tabs( 'tabs_btn_colors' );

        $this->start_controls_tab(
            'tab_btn_normal_colors',
            [
                'label' => esc_html__( 'Normal', 'avas-core' ),
            ]
        );

        $this->add_control(
            'btn_text_color',
            [
                'type' => Controls_Manager::COLOR,
                'label' => esc_html__( 'Text Color', 'avas-core' ),
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'btn_bg_color',
            [
                'type' => Controls_Manager::COLOR,
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'default' => '#605BE5',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'btn_border_color',
            [
                'type' => Controls_Manager::COLOR,
                'label' => esc_html__( 'Border Color', 'avas-core' ),
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'btn_box_shadow',
                'selector' => '{{WRAPPER}} .tx-search-form-submit',
                'condition' => [
                    'search_btn_style' => 'outer',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_btn_hover_colors',
            [
                'label' => esc_html__( 'Hover', 'avas-core' ),
            ]
        );


        $this->add_control(
            'btn_hv_text_color',
            [
                'type' => Controls_Manager::COLOR,
                'label' => esc_html__( 'Text Color', 'avas-core' ),
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'btn_hv_bg_color',
            [
                'type' => Controls_Manager::COLOR,
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'default' => '#4A45D2',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'btn_hv_border_color',
            [
                'type' => Controls_Manager::COLOR,
                'label' => esc_html__( 'Border Color', 'avas-core' ),
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit:hover' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'btn_hv_box_shadow',
                'selector' => '{{WRAPPER}} .tx-search-form-submit:hover',
                'condition' => [
                    'search_btn_style' => 'outer',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_responsive_control(
            'btn_width',
            [
                'label' => esc_html__( 'Width', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 125,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit' => 'min-width: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'btn_height',
            [
                'label' => esc_html__( 'Height', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 50,
                ],
                'selectors' => [
                    '{{WRAPPER}}.tx-search-form-style-outer .tx-search-form-submit' => 'height: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'search_btn_style' => 'outer',
                ],
            ]
        );

        $this->add_control(
            'btn_gutter',
            [
                'type' => Controls_Manager::SLIDER,
                'label' => esc_html__( 'Gutter', 'avas-core' ),
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}}.tx-search-form-style-outer.tx-search-form-position-right .tx-search-form-submit' => 'margin-left: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}}.tx-search-form-style-outer.tx-search-form-position-left .tx-search-form-submit' => 'margin-right: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'search_btn_style' => 'outer',
                ],
            ]
        );

        $this->add_control(
            'btn_position',
            [
                'label' => esc_html__( 'Position', 'avas-core' ),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'default' => 'right',
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'avas-core' ),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'avas-core' ),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'prefix_class' => 'tx-search-form-position-',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'btn_typography_divider',
            [
                'type' => Controls_Manager::DIVIDER,
                'style' => 'thick',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'btn_typography',
                'label' => esc_html__( 'Typography', 'avas-core' ),
                'selector' => '{{WRAPPER}} .tx-search-form-submit',
            ]
        );

        $this->add_control(
            'btn_border_size',
            [
                'label' => esc_html__( 'Border Size', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'btn_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-form-submit' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'separator' => 'after',
            ]
        );

        $this->end_controls_section();

        // Styles
        // Section: AJAX ------------
        // $this->add_section_style_ajax();

        $this->start_controls_section(
            'section_style_ajax',
            [
                'label' => esc_html__( 'Ajax', 'avas-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'ajax_search' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'heading_list',
            [
                'label' => esc_html__( 'Search List', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'background_color',
            [
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#FFFFFF',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'background_color_hover',
            [
                'label' => esc_html__( 'Background Color (Hover)', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#F6F6F6',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul li:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'ajax_box_shadow',
                'selector' => '{{WRAPPER}} .tx-search-plus-data-fetch'
            ]
        );

        $this->add_control(
            'search_list_item_transition_duration',
            [
                'label' => esc_html__( 'Transition Duration', 'avas-core' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 0.5,
                'min' => 0,
                'max' => 5,
                'step' => 0.1,
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul li' => 'transition-duration: {{VALUE}}s',
                ],
            ]
        );

        $this->add_responsive_control(
            'slider_content_hr',
            [
                'label' => esc_html__( 'Horizontal Position', 'avas-core' ),
                'type' => Controls_Manager::CHOOSE,
                'label_block' => false,
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'avas-core' ),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'avas-core' ),
                        'icon' => 'eicon-h-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'avas-core' ),
                        'icon' => 'eicon-h-align-right',
                    ]
                ],
                'selectors_dictionary' => [
                    'left' => 'left: 0; right: auto;',
                    'center' => 'left: 50%; transform: translateX(-50%)',
                    'right' => 'right: 0; left: auto;'
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => '{{VALUE}};',
                ],
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'search_list_width',
            [
                'label' => esc_html__( 'Container Width', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'vw'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 2000,
                    ],
                    '%' => [
                        'min' => 50,
                        'max' => 200,
                    ],
                    'vw' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'search_list_max_height',
            [
                'label' => esc_html__( 'Max Height', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'vh'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'vh',
                    'size' => 50,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul' => 'max-height: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'search_list_top_distance',
            [
                'label' => esc_html__( 'Top Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'vh'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 10,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'margin-top: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'list_item_title',
            [
                'label' => esc_html__( 'List Item', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'search_list_item_bottom_distance',
            [
                'label' => esc_html__( 'Bottom Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'vh'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 10,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul li:not(:last-child)' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_responsive_control(
            'search_list_item_padding',
            [
                'label' => esc_html__( 'Padding', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul li' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'heading_title',
            [
                'label' => esc_html__( 'Title', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#222222',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch a.tx-ajax-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title_typography',
                'selector' => '{{WRAPPER}} .tx-search-plus-data-fetch a.tx-ajax-title',
            ]
        );

        $this->add_responsive_control(
            'title_distance',
            [
                'label' => esc_html__( 'Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 2,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-content a.tx-ajax-title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'heading_description',
            [
                'label' => esc_html__( 'Description', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#757575',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch p a' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .tx-search-admin-notice' => 'color: {{VALUE}};'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description_typography',
                'selector' => '{{WRAPPER}} .tx-search-plus-data-fetch p a, {{WRAPPER}} .tx-search-admin-notice',
                'fields_options' => [
                    'typography'      => [
                        'default' => 'custom',
                    ],
                    'font_size'       => [
                        'default'    => [
                            'size' => '14',
                            'unit' => 'px',
                        ],
                    ]
                ],
            ]
        );

        $this->add_responsive_control(
            'description_distance',
            [
                'label' => esc_html__( 'Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 2,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-content p.tx-ajax-desc' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'heading_image',
            [
                'label' => esc_html__( 'Image', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
                'condition' => [
                    'show_ajax_thumbnails' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'image_width',
            [
                'label' => esc_html__( 'Width', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 300,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 150,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch a.tx-ajax-img-wrap' => 'width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-ajax-search-content' => 'width: calc(100% - {{SIZE}}{{UNIT}});',
                ],
                'condition' => [
                    'show_ajax_thumbnails' => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'image_distance',
            [
                'label' => esc_html__( 'Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch a.tx-ajax-img-wrap' => 'margin-right: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_ajax_thumbnails' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'view_result_text_heading',
            [
                'label' => esc_html__( 'View Result', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'view_result_text_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#FFFFFF',
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'view_result_text_color_hr',
            [
                'label' => esc_html__( 'Color (Hover)', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#FFFFFF',
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result:hover' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'view_result_text_bg_color',
            [
                'label' => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#605BE5',
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'view_result_text_bg_color_hr',
            [
                'label' => esc_html__( 'Background Color (Hover)', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#605BE5',
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'view_result_typography',
                'selector' => '{{WRAPPER}} a.tx-view-result',
                'fields_options' => [
                    'typography' => [
                        'default' => 'custom',
                    ],
                    'font_family' => [
                        'default' => 'Roboto',
                    ],
                    'font_size'   => [
                        'default' => [
                            'size' => '14',
                            'unit' => 'px',
                        ]
                    ]
                ]
            ]
        );

        $this->add_control(
            'view_result_transition_duration',
            [
                'label' => esc_html__( 'Transition Duration', 'avas-core' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 0.5,
                'min' => 0,
                'max' => 5,
                'step' => 0.1,
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result' => 'transition-duration: {{VALUE}}s',
                ],
            ]
        );

        $this->add_control(
            'view_result_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                ],
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]
        );

        $this->add_responsive_control(
            'view_result_padding',
            [
                'label' => esc_html__( 'Padding', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 5,
                    'right' => 10,
                    'bottom' => 5,
                    'left' => 10,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} a.tx-view-result' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'heading_close_btn',
            [
                'label' => esc_html__( 'Close Button', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'close_btn_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-search-plus-close' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'close_btn_size',
            [
                'label' => esc_html__( 'Size', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 14,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-search-plus-close::before' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-search-plus-close' => 'height: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_responsive_control(
            'close_btn_position',
            [
                'label' => esc_html__( 'Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 50,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-search-plus-close' => 'top: {{SIZE}}{{UNIT}}; right: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'scrollbar_heading',
            [
                'label' => esc_html__( 'Scrollbar', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'scrollbar_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul::-webkit-scrollbar-thumb' => 'border-left-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'scrollbar_width',
            [
                'label' => esc_html__( 'Width', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 10,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 3,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul::-webkit-scrollbar-thumb' => 'border-left-width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul::-webkit-scrollbar' => 'width: calc({{SIZE}}{{UNIT}} + 3px);',
                ]
            ]
        );

        $this->add_control(
            'no_results_heading',
            [
                'label' => esc_html__( 'No Results', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'no_results_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#222222',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-no-results' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'no_results_typography',
                'selector' => '{{WRAPPER}} .tx-search-plus-data-fetch .tx-no-results',
            ]
        );

        $this->add_responsive_control(
            'no_results_height',
            [
                'label' => esc_html__( 'Height', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'vh'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                    ],
                    'vh' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'vh',
                    'size' => 20,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch .tx-no-results' => 'height: {{SIZE}}{{UNIT}};',
                ]
            ]
        );

        $this->add_control(
            'search_results_box_border_size',
            [
                'label' => esc_html__( 'Border Size', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 1,
                    'right' => 1,
                    'bottom' => 1,
                    'left' => 1,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'search_results_box_border_radius',
            [
                'label' => esc_html__( 'Border Radius', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                ],
                'size_units' => [ 'px', '%' ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'separator' => 'after',
            ]
        );

        $this->add_responsive_control(
            'search_results_box_padding',
            [
                'label' => esc_html__( 'Padding', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'default' => [
                    'top' => 2,
                    'right' => 2,
                    'bottom' => 2,
                    'left' => 2,
                ],
                'size_units' => [ 'px', ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-plus-data-fetch ul' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'price_heading',
            [
                'label' => esc_html__( 'Price', 'avas-core' ),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'product_price_color',
            [
                'label' => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#666666',
                'selectors' => [
                    '{{WRAPPER}} .tx-search-product-price' => 'color: {{VALUE}};',
                ],
                'condition' => [
                    'show_product_price' => 'yes',
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'product_price_spacing',
            [
                'label' => esc_html__( 'Distance', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 25,
                    ]
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 5,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-search-product-price' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_product_price' => 'yes',
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // Styles ====================
        // Section: Pagination -------
        $this->start_controls_section(
            'section_style_pagination',
            [
                'label' => esc_html__( 'AJAX Pagination', 'avas-core' ),
                'tab' => Controls_Manager::TAB_STYLE,
                'show_label' => false,
                'condition' => [
                    'ajax_search' => 'yes'
                ]
            ]
        );

        $this->start_controls_tabs( 'tabs_grid_pagination_style' );

        $this->start_controls_tab(
            'tab_grid_pagination_normal',
            [
                'label' => esc_html__( 'Normal', 'avas-core' ),
            ]
        );

        $this->add_control(
            'pagination_color',
            [
                'label'  => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .tx-ajax-search-pagination svg' => 'fill: {{VALUE}}',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'color: {{VALUE}}'
                ],
            ]
        );

        $this->add_control(
            'pagination_bg_color',
            [
                'label'  => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#605BE5',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-no-more-results' => 'color: {{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'pagination_border_color',
            [
                'label'  => esc_html__( 'Border Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'border-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'border-color: {{VALUE}}'
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'pagination_box_shadow',
                'selector' => '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results, {{WRAPPER}} .tx-ajax-search-pagination > div > span',
            ]
        );

        $this->add_control(
            'pagination_loader_color',
            [
                'label'  => esc_html__( 'Loader Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#605BE5',
                'selectors' => [
                    '{{WRAPPER}} .tx-double-bounce .tx-child' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-wave .tx-rect' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-spinner-pulse' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-chasing-dots .tx-child' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-three-bounce .tx-child' => 'background-color: {{VALUE}}',
                    '{{WRAPPER}} .tx-fading-circle .tx-circle:before' => 'background-color: {{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'pagination_wrapper_color',
            [
                'label'  => esc_html__( 'Wrapper Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination' => 'background-color: {{VALUE}}',
                ],
                'separator' => 'after',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_grid_pagination_hover',
            [
                'label' => esc_html__( 'Hover', 'avas-core' ),
            ]
        );

        $this->add_control(
            'pagination_color_hr',
            [
                'label'  => esc_html__( 'Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#ffffff',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results:hover' => 'color: {{VALUE}}',
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results:hover svg' => 'fill: {{VALUE}}',
                ],
            ]
        );

        $this->add_control(
            'pagination_bg_color_hr',
            [
                'label'  => esc_html__( 'Background Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#4A45D2',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results:hover' => 'background-color: {{VALUE}}',
                ]
            ]
        );

        $this->add_control(
            'pagination_border_color_hr',
            [
                'label'  => esc_html__( 'Border Color', 'avas-core' ),
                'type' => Controls_Manager::COLOR,
                'default' => '#E8E8E8',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results:hover' => 'border-color: {{VALUE}}',
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'pagination_box_shadow_hr',
                'selector' => '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results:hover',
                'separator' => 'after',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->add_control(
            'pagination_transition_duration',
            [
                'label' => esc_html__( 'Transition Duration', 'avas-core' ),
                'type' => Controls_Manager::NUMBER,
                'default' => 0.1,
                'min' => 0,
                'max' => 5,
                'step' => 0.1,
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'transition-duration: {{VALUE}}s',
                    '{{WRAPPER}} .tx-ajax-search-pagination svg' => 'transition-duration: {{VALUE}}s',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'transition-duration: {{VALUE}}s',
                ],
                'separator' => 'after',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'pagination_typography',
                'selector' => '{{WRAPPER}} .tx-ajax-search-pagination, {{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results'
            ]
        );

        $this->add_control(
            'pagination_border_type',
            [
                'label' => esc_html__( 'Border Type', 'avas-core' ),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'none' => esc_html__( 'None', 'avas-core' ),
                    'solid' => esc_html__( 'Solid', 'avas-core' ),
                    'double' => esc_html__( 'Double', 'avas-core' ),
                    'dotted' => esc_html__( 'Dotted', 'avas-core' ),
                    'dashed' => esc_html__( 'Dashed', 'avas-core' ),
                    'groove' => esc_html__( 'Groove', 'avas-core' ),
                ],
                'default' => 'none',
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'border-style: {{VALUE}};',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'border-style: {{VALUE}};'
                ],
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'pagination_border_width',
            [
                'label' => esc_html__( 'Border Width', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'default' => [
                    'top' => 1,
                    'right' => 1,
                    'bottom' => 1,
                    'left' => 1,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'condition' => [
                    'pagination_border_type!' => 'none',
                ],
            ]
        );

        $this->add_responsive_control(
            'pagination_distance_from_grid',
            [
                'label' => esc_html__( 'Distance From List', 'avas-core' ),
                'type' => Controls_Manager::SLIDER,
                'size_units' => [ 'px' ],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],              
                'default' => [
                    'unit' => 'px',
                    'size' => 25,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'pagination_padding',
            [
                'label' => esc_html__( 'Padding', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'default' => [
                    'top' => 8,
                    'right' => 20,
                    'bottom' => 8,
                    'left' => 20,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'pagination_wrapper_padding',
            [
                'label' => esc_html__( 'Wrapper Padding', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px' ],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'pagination_radius',
            [
                'label' => esc_html__( 'Border Radius', 'avas-core' ),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => [ 'px', '%' ],
                'default' => [
                    'top' => 0,
                    'right' => 0,
                    'bottom' => 0,
                    'left' => 0,
                ],
                'selectors' => [
                    '{{WRAPPER}} .tx-ajax-search-pagination .tx-load-more-results' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .tx-ajax-search-pagination > div > span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ],
                'separator' => 'before',
            ]
        );

        $this->end_controls_section();
        
    }

    public function render_search_pagination($settings) {
        if ( 'yes' === $settings['ajax_search'] ) :

            echo '<div class="tx-ajax-search-pagination elementor-clearfix tx-ajax-search-pagination-load-more">';
            echo '<button class="tx-load-more-results">'. esc_html__($settings['pagination_load_more_text'], 'avas-core') .'</button>';
            echo '<div class="tx-pagination-loading">';
                switch ( $settings['pagination_animation'] ) {
                    case 'loader-1':
                        echo '<div class="tx-loader-1"></div>';
                        break;
                    case 'loader-2':
                        echo '<div class="tx-loader-2"></div>';
                        break;
                    case 'loader-3':
                        echo '<div class="tx-loader-3"></div>';
                        break;
                    default:
                        break;
                }
            echo '</div>';

            echo '<p class="tx-no-more-results">'. esc_html($settings['pagination_finish_text']) .'</p>';

            echo '</div>';

        endif;
    }

    protected function render_search_submit_btn() {
        $settings = $this->get_settings_for_display();

        $this->add_render_attribute(
        'button', [
            'class' => 'tx-search-form-submit',
            'aria-label' => $settings['search_aria_label'],
            'type' => 'submit',
        ]
        );

        if ( $settings['search_btn_disable_click'] ) {
            $this->add_render_attribute( 'button', 'disabled' );
        }

        if ( 'yes' === $settings['search_btn'] ) : ?>

        <button <?php echo $this->get_render_attribute_string( 'button' ); ?>>
            <?php if ( 'icon' === $settings['search_btn_type'] && '' !== $settings['search_btn_icon']['value'] ) : ?>
                <i class="<?php echo esc_attr( $settings['search_btn_icon']['value'] ); ?>"></i>
            <?php elseif( 'text' === $settings['search_btn_type'] && '' !== $settings['search_btn_text'] ) : ?>
                <?php echo esc_html( $settings['search_btn_text'] ); ?>
            <?php endif; ?>
        </button>

        <?php
        endif;
    }

    // Recursive function to print categories and their children
    protected function print_categories($category, $taxonomy, $level = 0) {
        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
            'parent' => $category->term_id, // get children of this category
        );

        $children = get_terms($args);

        $output = '';

        if ( 'uncategorized' !== $category->slug ) {
            $output = '<option value="' . $category->term_id . '" data-post-type="'. $category->taxonomy .'">' . str_repeat('&nbsp;', $level * 3) . $category->name . '</option>';
        }

        foreach ($children as $child) {
            $output .= $this->print_categories($child, $taxonomy, $level + 1);
        }

        return $output;
    }

    protected function render_categories_select($post_type, $all_cat_text, $taxonomy) {
        // Define taxonomy based on post_type
        if ( $post_type == 'all' ) {
            $post_types = ['post', 'product'];
            if ( class_exists('WooCommerce') ) {
                $post_types = ['post', 'product'];
            } else {
                $post_types = ['post'];
            }

            $output = '<div class="tx-category-select-wrap">';
            
            // Start the select
            $output .= '<select name="category" class="tx-category-select">';

            $output .= '<option value="0">'. $all_cat_text .'</option>';
    
            // Fetch and add options for each post type
            foreach ($post_types as $post_type) {
                $taxonomy = ($post_type == 'product') ? 'product_cat' : 'category';
        
                $args = array(
                    'taxonomy' => $taxonomy,
                    'hide_empty' => false,
                    'parent' => 0, // get top level categories
                );
                
                $categories = get_terms($args);
        
                foreach ($categories as $category) {
                    $output .= $this->print_categories($category, $taxonomy);
                }

            }
        
            // Close the select
            $output .= '</select>';
            $output .= '</div>';
        
            // Return the generated select
            return $output;
        } else {
            $taxonomy = $taxonomy;
    
            // Prepare the arguments for wp_dropdown_categories
            $args = array(
                'show_option_all'    => $all_cat_text,
                'show_option_none'   => '',
                'option_none_value'  => '-1',
                'orderby'            => 'name',
                'order'              => 'ASC',
                'show_count'         => 0,
                'hide_empty'         => 0,
                'child_of'           => 0,
                'include'            => '',
                'echo'               => 0,
                'selected'           => 0,
                'hierarchical'       => 1, 
                'name'               => 'category',
                'class'              => 'tx-category-select',
                'depth'              => 0,
                'tab_index'          => 0,
                'taxonomy'           => $taxonomy,
                'hide_if_empty'      => false,
                'value_field'        => 'term_id',   
            );

            if ( 'product_cat' == $taxonomy ) {
                $uncategorized_product_cat = get_term_by('slug', 'uncategorized', 'product_cat');
                $exclude_id = ($uncategorized_product_cat) ? $uncategorized_product_cat->term_id : '';
                $args['exclude'] = $exclude_id;
            } else {
                $args['exclude'] = 1;
            }

            $select_name = 'tx_select_category';

            if ( $post_type == 'product' ) {
                $select_name = 'tx_select_product_cat';
            }


            $output = '<div class="tx-category-select-wrap">';
            
            // Start the select
            $output .= '<select name="'. $select_name .'" class="tx-category-select">';

            $output .= '<option value="0">'. $all_cat_text .'</option>';
    
            // Fetch and add options for each post
                if ( is_array($taxonomy) && count($taxonomy) > 1) {
                    $categories = [];
                    $taxonomies = [];

                    foreach($taxonomy as $taxonomy_type) {

                        $args = array(
                            'taxonomy' => $taxonomy_type,
                            'hide_empty' => false,
                            'parent' => 0, // get top level categories
                        );
                        
                        $taxonomies = get_terms($args);
                        $categories = $taxonomies;
                        
                        // Group options by post type
                        $output .= '<optgroup label="' . ucfirst(get_taxonomy($taxonomy_type)->label) . '">';
                
                        foreach ($categories as $category) {
                            $output .= $this->print_categories($category, $taxonomy);
                        }
                
                        $output .= '</optgroup>';
                    }
                } else {
                    $args = array(
                        'taxonomy' => $taxonomy,
                        'hide_empty' => false,
                        'parent' => 0, // get top level categories
                    );

                    $categories = get_terms($args);
        
                    foreach ($categories as $category) {
                        $output .= $this->print_categories($category, $taxonomy);
                    }
                }
        
            // Close the select
            $output .= '</select>';
            $output .= '</div>';
        
            // Return the generated select
            return $output;
        }
    }   
    
    protected function render() {
        // Get Settings
        $settings = $this->get_settings_for_display();

        $hidden_input = '';

        $this->add_render_attribute(
            'input', [
                'class' => 'tx-search-form-input',
                'placeholder' => $settings['search_placeholder'],
                // 'data-exclude-titles' => esc_attr( $settings['exclude_titles'] ),
                'aria-label' => $settings['search_aria_label'],
                'type' => 'search',
                'name' => 's',
                'title' => esc_html__( 'Search', 'avas-core' ),
                'value' => get_search_query(),
                'tx-query-type' => $settings['search_query'],
                'tx-taxonomy-type' => isset($settings['query_taxonomy_'. $settings['search_query']]) ? $settings['query_taxonomy_'. $settings['search_query']] : '',
                'number-of-results' => isset($settings['number_of_results']) ? $settings['number_of_results'] : -1,
                'ajax-search' => isset($settings['ajax_search']) ? $settings['ajax_search'] : '',
                'show-description' => isset($settings['show_description']) ? $settings['show_description'] : '',
                'number-of-words' => isset($settings['number_of_words_in_excerpt']) ? $settings['number_of_words_in_excerpt'] : '',
                'show-ajax-thumbnails' => isset($settings['show_ajax_thumbnails']) ? $settings['show_ajax_thumbnails'] : '',
                'show-view-result-btn' => isset($settings['show_view_result_btn']) ? $settings['show_view_result_btn'] : '',
                'show-product-price' => isset($settings['show_product_price']) ? $settings['show_product_price'] : '',
                'view-result-text' => isset($settings['show_ajax_thumbnails']) ? $settings['view_result_text'] : '',
                'no-results' => isset($settings['no_results_text']) ? esc_html__($settings['no_results_text']) : '',
                'exclude-without-thumb' => isset($settings['exclude_posts_without_thumbnail']) ? $settings['exclude_posts_without_thumbnail'] : '',
                'link-target' => isset($settings['ajax_search_link_target']) && ( 'yes' === $settings['ajax_search_link_target'] ) ? '_blank'  : '_self',
                'password-protected' => isset($settings['ajax_show_ps_pt']) ? $settings['ajax_show_ps_pt'] : 'no',  
                'attachments' => isset($settings['show_attachments']) ? $settings['show_attachments'] : 'no',
            ]
        );

        // add this attribute
        $this->add_render_attribute(
            'input',
            'data-exclude-ids',
            ! empty( $settings['exclude_posts'] )
                ? implode( ',', $settings['exclude_posts'] )
                : ''
        );

        ?>

        <form role="search" method="get" class="tx-search-form" action="<?php echo home_url(); ?>">
            <?php if ( 'all' !== $settings['search_query'] ) : ?>
                <input type="hidden" name="post_type" value="<?php echo esc_attr( $settings['search_query'] ); ?>" />
            <?php endif; ?>

            <div class="tx-search-form-input-wrap elementor-clearfix">
                <?php

                    if ( 'all' !== $settings['search_query'] && 'yes' == $settings['select_category'] ) {
                        $post_type = $settings['search_query'];
                        $selected_taxonomy = isset($settings['query_taxonomy_' . $post_type]) && !empty($settings['query_taxonomy_' . $post_type])
                            ? $settings['query_taxonomy_' . $post_type]
                            : '';

                        if (empty($selected_taxonomy)) {
                            $taxonomies = get_object_taxonomies($post_type, 'objects');
                            if (!empty($taxonomies)) {
                                foreach ($taxonomies as $tax) {
                                    if ($tax->hierarchical) {
                                        $selected_taxonomy = $tax->name;
                                        break;
                                    }
                                }
                                if (empty($selected_taxonomy)) {
                                    $selected_taxonomy = reset($taxonomies)->name;
                                }
                            } else {
                                $selected_taxonomy = false;
                            }
                        }

                        if ($selected_taxonomy) {
                            echo $this->render_categories_select($post_type, esc_html($settings['all_cat_text']), $selected_taxonomy);
                        }
                    }
                ?>
                <input <?php echo $this->get_render_attribute_string( 'input' ); ?>>
                <?php
                if ( $settings['search_btn_style'] === 'inner' ) {
                    $this->render_search_submit_btn();
                }
                ?>
            </div>

            <?php

            if ( $settings['search_btn_style'] === 'outer' ) {
                $this->render_search_submit_btn();
            }

            ?>
            
        </form>
        <div class="tx-search-plus-data-fetch">
            <span class="tx-search-plus-close"></span>
            <ul></ul>
            <?php echo $this->render_search_pagination($settings); ?>
        </div>
        
        <?php

    }

}