<?php

namespace AvasElements\Modules\Conditions;

use AvasElements\Base\Module_Base;
use Elementor\Element_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

class Module extends Module_Base {

     /**
     * Initialize hooks
     */
    public function __construct() {

        // Register the controls in Elementor
        add_action( 'elementor/element/common/_section_style/after_section_end', [ $this, 'register_controls' ] );
        add_action( 'elementor/element/column/section_advanced/after_section_end', [ $this, 'register_controls' ] );
        add_action( 'elementor/element/section/section_advanced/after_section_end', [ $this, 'register_controls' ] );
        add_action( 'elementor/element/container/section_layout/after_section_end', [ $this, 'register_controls' ] );

        //Add custom classes for the display condition
        add_action('elementor/frontend/container/before_render', [$this, 'add_data_attributes'], 10, 1);
        add_action('elementor/frontend/section/before_render', [$this, 'add_data_attributes'], 10, 1);
        add_action('elementor/frontend/widget/before_render', [$this, 'add_data_attributes'], 10, 1);

        // Enqueue the JS script when Elementor is rendering
        add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'enqueue_conditional_display_script' ] );

    }

    public function get_name() {
        return 'avas-conditions';
    }

    /**
     * Enqueue the custom jQuery script
     */
    public function enqueue_conditional_display_script() {
        wp_enqueue_script('tx-conditions');

        // Pass front page, 404 page, and archive page status to the JavaScript
        wp_localize_script('tx-conditions', 'TxConditionalDisplay', [
            'isFrontPage' => is_front_page(),
            'is404Page' => is_404(),
            'isArchivePage' => is_archive(),
        ]);

    }

    /**
     * Register controls in Elementor for conditional display.
     *
     * @param Element_Base $element Elementor element instance.
     */
    public function register_controls( $element ) {
        $element->start_controls_section(
            'custom_conditional_display',
            [
                'label' =>  esc_html__( 'Avas Conditions', 'avas-core' ),
                'tab'   => Controls_Manager::TAB_ADVANCED,
            ]
        );

        $element->add_control(
            'custom_enable_condition',
            [
                'label'        => esc_html__( 'Enable Conditional Display', 'avas-core' ),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => esc_html__( 'Yes', 'avas-core' ),
                'label_off'    => esc_html__( 'No', 'avas-core' ),
                'return_value' => 'yes',
                'default'      => '',
            ]
        );

        // Repeater field for adding multiple conditions
        $repeater = new Repeater();

        $repeater->add_control(
            'page_condition',
            [
                'label'     => esc_html__( 'Condition', 'avas-core' ),
                'type'      => Controls_Manager::SELECT,
                'options'   => [
                    'front'   => esc_html__( 'Front Page', 'avas-core' ),
                    'others'  => esc_html__( 'Other Pages', 'avas-core' ),
                    '404'     => esc_html__( '404 Page', 'avas-core' ),
                    'archive' => esc_html__( 'Archive Page', 'avas-core' ),
                ],
                'default'   => 'front',
            ]
        );

        $repeater->add_control(
            'visibility_action',
            [
                'label'     => esc_html__( 'Action', 'avas-core' ),
                'type'      => Controls_Manager::SELECT,
                'options'   => [
                    'show' => esc_html__( 'Show', 'avas-core' ),
                    'hide' => esc_html__( 'Hide', 'avas-core' ),
                ],
                'default'   => 'show',
            ]
        );

        $element->add_control(
            'conditions_list',
            [
                'label'       => esc_html__( 'Conditions', 'avas-core' ),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'title_field' => '<# print(page_condition.replace(/_/g, " ").split(" ").map(word => word.charAt(0).toUpperCase() + word.slice(1)).join(" ") + " - " + visibility_action.charAt(0).toUpperCase() + visibility_action.slice(1)); #>',
                'condition'   => [
                    'custom_enable_condition' => 'yes',
                ],
            ]
        );

        $element->end_controls_section();
    }


    /**
     * Add custom classes based on the condition selected.
     *
     * @param Element_Base $element Elementor element instance.
     */
    public function add_data_attributes( $element ) {
        $settings = $element->get_settings_for_display();

        if ( isset( $settings['custom_enable_condition'] ) && $settings['custom_enable_condition'] === 'yes' ) {
            if ( isset( $settings['conditions_list'] ) && is_array( $settings['conditions_list'] ) ) {
                $data_values = [];

                foreach ( $settings['conditions_list'] as $condition ) {
                    $page_condition = $condition['page_condition'];
                    $visibility_action = $condition['visibility_action'];

                    // Add the appropriate condition based on the user's selection
                    if ( $page_condition === 'front' ) {
                        $data_values[] = ( $visibility_action === 'show' ) ? 'show-on-front' : 'hide-on-front';
                    } elseif ( $page_condition === 'others' ) {
                        $data_values[] = ( $visibility_action === 'show' ) ? 'show-on-others' : 'hide-on-others';
                    } elseif ( $page_condition === '404' ) {
                        $data_values[] = ( $visibility_action === 'show' ) ? 'show-on-404' : 'hide-on-404';
                    } elseif ( $page_condition === 'archive' ) {
                        $data_values[] = ( $visibility_action === 'show' ) ? 'show-on-archive' : 'hide-on-archive';
                    }
                }

                // Join all conditions with a space separator and add it as a single data attribute
                if ( !empty( $data_values ) ) {
                    $element->add_render_attribute( '_wrapper', 'tx-data-display-condition', implode( ' ', $data_values ) );
                }
            }
        }
    }


}
