<?php
namespace AvasElements\Modules\CircleText\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Css_Filter;
use Elementor\Utils;
use AvasElements\TX_Helper;

if ( ! defined( 'ABSPATH' ) ) exit;

class CircleText extends Widget_Base {

    public function get_name() {
        return 'avas-circle-text';
    }

    public function get_title() {
        return esc_html__( 'Avas Circle Text', 'avas-core' );
    }

    public function get_icon() {
        return 'eicon-circle';
    }

    public function get_style_depends() {
        return [ 'tx-circle-text' ];
    }

    public function get_script_depends() {
        return [ 'tx-circle-text' ];
    }

    public function get_categories() {
        return [ 'tx-elements' ];
    }

    public function get_keywords() {
        return [ 'circle', 'text', 'rotate', 'logo' ];
    }

    protected function register_controls() {

        // Content Controls
        $this->start_controls_section('content_section', [
            'label' => esc_html__('Content', 'avas-core'),
        ]);

        $this->add_control('tx_circle_text', [
            'label'   => esc_html__('Text', 'avas-core'),
            'type'    => Controls_Manager::TEXTAREA,
            'dynamic' => ['active' => true],
            'default' => 'Some Text - Animated Circle Text -',
        ]);

        $this->add_control('tx_circle_text_html_tag', [
            'label'   => esc_html__('HTML Tag', 'avas-core'),
            'type'    => Controls_Manager::SELECT,
            'default' => 'p',
            'options' => TX_Helper::title_html_tags(),
        ]);

        $this->add_control('tx_circle_text_image', [
            'label'   => esc_html__('Image', 'avas-core'),
            'type'    => Controls_Manager::MEDIA,
            'default' => ['url' => Utils::get_placeholder_image_src()],
        ]);

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name'    => 'tx_circle_text_image',
                'default' => 'thumbnail',
                'condition' => [
                    'tx_circle_text_image[url]!' => '',
                ],
            ]
        );

        $this->add_control('tx_ct_circle_size', [
            'label' => esc_html__('Circle Size (px)', 'avas-core'),
            'type' => Controls_Manager::SLIDER,
            'size_units' => ['px'],
            'range' => [
                'px' => [
                    'min' => 100,
                    'max' => 600,
                    'step' => 1,
                ],
            ],
            'default' => [
                'unit' => 'px',
                'size' => 200,
            ],
        ]);

        $this->add_control('tx_ct_animation_speed', [
            'label' => esc_html__('Rotation Speed (sec)', 'avas-core'),
            'type' => Controls_Manager::SLIDER,
            'range' => [
                'px' => [
                    'min' => 1,
                    'max' => 60,
                    'step' => 1,
                ],
            ],
            'default' => [
                'unit' => 'px',
                'size' => 15,
            ],
        ]);

        $this->add_control('tx_ct_rotate_direction', [
            'label'   => esc_html__('Rotation Direction', 'avas-core'),
            'type'    => Controls_Manager::SELECT,
            'default' => 'normal',
            'options' => [
                'normal'  => esc_html__('Left to Right (Clockwise)', 'avas-core'),
                'reverse' => esc_html__('Right to Left (Counter-Clockwise)', 'avas-core'),
            ],
        ]);

        $this->end_controls_section();

        // Style Controls
        $this->start_controls_section('style_section', [
            'label' => esc_html__('Text Style', 'avas-core'),
            'tab'   => Controls_Manager::TAB_STYLE,
        ]);

        $this->add_control('text_color', [
            'label' => esc_html__('Text Color', 'avas-core'),
            'type'  => Controls_Manager::COLOR,
            'selectors' => [
                '{{WRAPPER}} .tx-circle-text-txt' => 'color: {{VALUE}}',
            ],
        ]);

        $this->add_group_control(Group_Control_Typography::get_type(), [
            'name'     => 'typography',
            'label'    => esc_html__('Typography', 'avas-core'),
            'selector' => '{{WRAPPER}} .tx-circle-text-txt',
        ]);

        $this->add_group_control(
            Group_Control_Css_Filter::get_type(),
            [
                'name' => 'circle_text__css_filters',
                'selector' => '{{WRAPPER}} .tx-circle-text-image',
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings     = $this->get_settings_for_display();
        $text         = wp_kses_post($settings['tx_circle_text']);
        $tag          = esc_attr($settings['tx_circle_text_html_tag']);
        $direction    = ($settings['tx_ct_rotate_direction'] === 'reverse') ? 'tx-reverse' : '';
        $circle_size  = isset($settings['tx_ct_circle_size']['size']) ? intval($settings['tx_ct_circle_size']['size']) : 200;
        $speed        = isset($settings['tx_ct_animation_speed']['size']) ? floatval($settings['tx_ct_animation_speed']['size']) : 15;
        $logo_url = '';
        if ( ! empty( $settings['tx_circle_text_image']['id'] ) ) {
            $logo_url = Group_Control_Image_Size::get_attachment_image_src(
                $settings['tx_circle_text_image']['id'],
                'tx_circle_text_image',
                $settings
            );
        } elseif ( ! empty( $settings['tx_circle_text_image']['url'] ) ) {
            $logo_url = $settings['tx_circle_text_image']['url']; // fallback to URL (e.g., placeholder)
        }
        $logo = esc_url($logo_url);

        ?>
        <div class="tx-circle-text-wrap" style="width:<?= $circle_size ?>px; height:<?= $circle_size ?>px;">
            <div class="tx-circle-text-image" style="background-image:url('<?= $logo ?>'); width:<?= $circle_size * 0.7 ?>px; height:<?= $circle_size * 0.7 ?>px;"></div>
            <div class="tx-circle-text-txt <?= esc_attr($direction) ?>" style="animation-duration: <?= $speed ?>s; animation-timing-function: linear; animation-iteration-count: infinite;">
                <?php printf('<%1$s>%2$s</%1$s>', $tag, $text); ?>
            </div>
        </div>
        <?php
    }
}