<?php

namespace JNews\WEBSTORIES\Module;

/**
 * Class Module_Query
 *
 * @package  JNews\WEBSTORIES\Module
 */
class Module_Query {
	/**
	 * @var array
	 */
	private static $cache_thumbnail = array();

	/**
	 * @param $attr
	 *
	 * @return array|bool|mixed
	 */
	public static function webstories_query( $attr ) {
		$attr       = self::unset_unnecessary( $attr );
		$query_hash = 'query_hash_' . md5( serialize( $attr ) );

		if ( ! $result = wp_cache_get( $query_hash, 'jnews-webstories' ) ) {
			$result = self::default_query( $attr );
			// need to optimize query
			self::optimize_query( $result );
		}

		return $result;
	}

	/**
	 * @param $attr
	 *
	 * @return mixed
	 */
	private static function unset_unnecessary( $attr ) {
		$accepted = array(
			'post_type',
			'number_post',
			'post_offset',
			'include_post',
			'exclude_post',
			'include_category',
			'exclude_category',
			'include_author',
			'include_tag',
			'exclude_tag',
			'sort_by',
			'paged',
			'video_only',
			'content_type',
			'pagination_number_post',
			'pagination_mode',
			'date_query',
		);

		$accepted = apply_filters( 'jnews_unset_unnecessary_attr', $accepted, $attr );

		foreach ( $attr as $key => $value ) {
			if ( ! in_array( $key, $accepted, true ) ) {
				unset( $attr[ $key ] );
			}
		}

		if ( isset( $attr['pagination_number_post'] ) ) {
			$attr['pagination_number_post'] = (int) $attr['pagination_number_post'];
		}

		if ( isset( $attr['paged'] ) ) {
			$attr['paged'] = (int) $attr['paged'];
		}

		if ( isset( $attr['number_post']['size'] ) ) {
			$attr['number_post'] = $attr['number_post']['size'];
		}

		return $attr;
	}

	/**
	 * Calculate Offset
	 *
	 * @param $paged
	 * @param $offset
	 * @param $number_post
	 * @param $number_post_ajax
	 *
	 * @return int
	 */
	private static function calculate_offset( $paged, $offset, $number_post, $number_post_ajax ) {
		$new_offset = 0;

		if ( isset( $offset['size'] ) ) {
			$offset = $offset['size'];
		}

		if ( 1 == $paged ) {
			$new_offset = (int) $offset;
		}
		if ( 2 == $paged ) {
			$new_offset = $number_post + (int) $offset;
		}
		if ( $paged >= 3 ) {
			$new_offset = $number_post + (int) $offset + ( $number_post_ajax * ( $paged - 2 ) );
		}

		return $new_offset;
	}

	/**
	 * Check if we have next page
	 *
	 * @param $total
	 * @param int          $curpage
	 * @param int          $offset
	 * @param $perpage
	 * @param $perpage_ajax
	 *
	 * @return bool
	 */
	private static function has_next_page( $total, $perpage, $perpage_ajax, $curpage = 1, $offset = 0 ) {
		$flag = false;
		if ( 1 == $curpage ) {
			$flag = (int) $total > (int) $offset + (int) $perpage;
		} elseif ( $curpage > 1 ) {
			$flag = (int) $total > (int) $offset + (int) $perpage_ajax;
		}

		return $flag;
	}

	/**
	 * Check if we have previous page
	 *
	 * @param int $curpage
	 *
	 * @return bool
	 */
	private static function has_prev_page( $curpage = 1 ) {
		return ! ( $curpage <= 1 );
	}

	/**
	 * Get total count of total page
	 *
	 * @param $total
	 * @param int          $curpage
	 * @param int          $offset
	 * @param $perpage
	 * @param $perpage_ajax
	 *
	 * @return int
	 */
	private static function count_total_page( $total, $perpage, $perpage_ajax, $curpage = 1, $offset = 0 ) {
		$remain = (int) $total - ( (int) $offset + (int) $perpage );

		if ( $remain > 0 ) {
			while ( $remain > 0 ) {
				$remain -= (int) $perpage_ajax;
				++ $curpage;
			}
		}

		return $curpage;
	}

	/**
	 * WordPress Default Query
	 *
	 * @param $attr
	 *
	 * @return array
	 */
	private static function default_query( $attr ) {
		$args             = array();
		$result           = $args;
		$exclude_category = $result;
		$include_category = $exclude_category;

		$attr['number_post']            = isset( $attr['number_post'] ) ? $attr['number_post'] : get_option( 'posts_per_page' );
		$attr['pagination_number_post'] = isset( $attr['pagination_number_post'] ) ? $attr['pagination_number_post'] : $attr['number_post'];

		// Argument
		$args['post_type']           = isset( $attr['post_type'] ) ? $attr['post_type'] : 'web-story';
		$args['paged']               = isset( $attr['paged'] ) ? $attr['paged'] : 1;
		$args['offset']              = self::calculate_offset( $args['paged'], 0, $attr['number_post'], $attr['pagination_number_post'] );
		$args['posts_per_page']      = ( $args['paged'] > 1 ) ? $attr['pagination_number_post'] : $attr['number_post'];
		$args['no_found_rows']       = ! isset( $attr['pagination_mode'] ) || 'disable' === $attr['pagination_mode'];
		$args['ignore_sticky_posts'] = 1;

		if ( ! empty( $attr['include_post'] ) ) {
			$args['post__in'] = explode( ',', $attr['include_post'] );
		}

		if ( ! empty( $attr['exclude_post'] ) ) {
			$args['post__not_in'] = explode( ',', $attr['exclude_post'] );
		}

		if ( ! empty( $attr['include_category'] ) ) {
			$categories = explode( ',', $attr['include_category'] );
			self::recursive_category( $categories, $include_category );
			$args['web_story_category__in'] = $include_category;
		}

		if ( ! empty( $attr['exclude_category'] ) ) {
			$categories = explode( ',', $attr['exclude_category'] );
			self::recursive_category( $categories, $exclude_category );
			$args['web_story_category__not_in'] = $exclude_category;
		}

		if ( ! empty( $attr['include_tag'] ) ) {
			$args['web_story_tag__in'] = explode( ',', $attr['include_tag'] );
		}

		if ( ! empty( $attr['exclude_tag'] ) ) {
			$args['web_story_tag__not_in'] = explode( ',', $attr['exclude_tag'] );
		}

		// order
		if ( isset( $attr['sort_by'] ) ) {
			if ( 'latest' === $attr['sort_by'] ) {
				$args['orderby'] = 'date';
				$args['order']   = 'DESC';
			}

			if ( 'latest_modified' === $attr['sort_by'] ) {
				$args['orderby'] = 'modified';
				$args['order']   = 'DESC';
			}

			if ( 'oldest' === $attr['sort_by'] ) {
				$args['orderby'] = 'date';
				$args['order']   = 'ASC';
			}

			if ( 'oldest_modified' === $attr['sort_by'] ) {
				$args['orderby'] = 'modified';
				$args['order']   = 'ASC';
			}

			if ( 'alphabet_asc' === $attr['sort_by'] ) {
				$args['orderby'] = 'title';
				$args['order']   = 'ASC';
			}

			if ( 'alphabet_desc' === $attr['sort_by'] ) {
				$args['orderby'] = 'title';
				$args['order']   = 'DESC';
			}

			if ( 'random' === $attr['sort_by'] ) {
				$args['orderby'] = 'rand';
			}

			if ( 'random_week' === $attr['sort_by'] ) {
				$args['orderby']    = 'rand';
				$args['date_query'] = array(
					array(
						'after' => '1 week ago',
					),
				);
			}

			if ( 'random_month' === $attr['sort_by'] ) {
				$args['orderby']    = 'rand';
				$args['date_query'] = array(
					array(
						'after' => '1 month ago',
					),
				);
			}

			if ( 'share' === $attr['sort_by'] ) {
				$args['orderby']  = 'meta_value_num';
				$args['meta_key'] = 'jnews_social_counter_total';
				$args['order']    = 'DESC';
			}

			if ( 'post__in' === $attr['sort_by'] ) {
				$args['orderby'] = 'post__in';
			}
		}

		// date
		if ( isset( $attr['date_query'] ) ) {
			$args['date_query'] = $attr['date_query'];
		}

		if ( class_exists( 'Polylang' ) && isset( $attr['lang'] ) ) {
			$args['lang'] = $attr['lang'];
		}

		$args = apply_filters( 'jnews_default_query_webstories_args', $args, $attr );
		// Query

		$query = new \WP_Query( $args );

		foreach ( $query->posts as $post ) {
			$result[] = $post;
		}

		wp_reset_postdata();

		return array(
			'result'     => $result,
			'next'       => self::has_next_page( $query->found_posts, $attr['number_post'], $attr['pagination_number_post'], $args['paged'], $args['offset'] ),
			'prev'       => self::has_prev_page( $args['paged'] ),
			'total_page' => self::count_total_page( $query->found_posts, $attr['number_post'], $attr['pagination_number_post'], $args['paged'], $args['offset'] ),
		);
	}

		/**
		 * @param $categories
		 * @param $result
		 */
	private static function recursive_category( $categories, &$result ) {
		foreach ( $categories as $category ) {
			if ( ! in_array( $category, $result, true ) ) {
				$result[] = $category;
				$children = get_categories( array( 'parent' => $category ) );

				if ( ! empty( $children ) ) {
					$child_id = array();
					foreach ( $children as $child ) {
						$child_id[] = $child->term_id;
					}
					self::recursive_category( $child_id, $result );
				}
			}
		}
	}


	/**
	 * @param $result
	 */
	private static function optimize_query( $result ) {
		self::cache_thumbnail( $result );
	}

	/**
	 * @param $results
	 */
	public static function cache_thumbnail( $results ) {
		$thumbnails = array();

		foreach ( $results['result'] as $result ) {
			if ( ! in_array( $result->ID, self::$cache_thumbnail, true ) ) {
				$thumbnails[]            = get_post_thumbnail_id( $result->ID );
				self::$cache_thumbnail[] = $result->ID;
			}
		}

		if ( ! empty( $thumbnails ) ) {
			$query = array(
				'post__in'  => $thumbnails,
				'post_type' => 'attachment',
				'showposts' => count( $thumbnails ),
			);

			get_posts( $query );
		}
	}

	public function jnews_get_webstories( $args = '' ) {
		$webstories = get_posts(
			array(
				$args,
			)
		);
		return $webstories;
	}
}
